using System;
using System.Data;

using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{

	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>10/8/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>This class is used to parse and extract VistA Lab previous test results for a previously received VBECS order from CPRS.</summary>

	#endregion

	/// <summary>
	/// Class CprsProcessPreviousLabResults
	/// </summary>
	public class CprsProcessPreviousLabResults
	{
		private const string LAST_UPDATE_USER_PARAMETER_VALUE = "CPRS";
		private HL7OmgMessage _message;
		static CprsProcessPreviousLabResults() {}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5925"> 
		///		<ExpectedInput>Valid HL7OmgMessage Lab previous results message.</ExpectedInput>
		///		<ExpectedOutput>Non-null instance of CprgProcessPreviousLabResults.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5926"> 
		///		<ExpectedInput>Null HL7OmgMessage input parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor to initialize the CprsProcessPreviousLabResults instance and identify the HL7 message to process.
		/// </summary>
		/// <param name="message"></param>
		public CprsProcessPreviousLabResults( HL7OmgMessage message )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );

			this._message = message;
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5927"> 
		///		<ExpectedInput>Valid instance of CprsProcessPreviousLabResults type with previous results message.</ExpectedInput>
		///		<ExpectedOutput>Non-Null HL7OrgMessage with Acknowledgement code of 'AA' and an entry created in PatientTest
		///		with the Lab test result associated with the PatientOrder.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5928"> 
		///		<ExpectedInput>Invalid order control code of 'NW'.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5929"> 
		///		<ExpectedInput>HL7OmgMessage where the OrderGroupNumber does not exist in the VBECS database.</ExpectedInput>
		///		<ExpectedOutput>Application reject HL7 acknowledgement message</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method is responsible for processing the Previous Lab results HL7 message from CPRS and inserting the test result values in the VBECS database.
		/// </summary>
		/// <returns><see cref="HL7OrgMessage"/> containing a valid HL7 response message to the incoming previous results message.</returns>
		public HL7OrgMessage ProcessPreviousResults()
		{
			if( this.Message == null )
				throw( new ArgumentNullException( "_message" ) );

			if( _message.OrderControlCode != "PR" )
				throw( new HL7Exception( "Order Control Code must be 'PR' to call CprsProcessPreviousLabResults.ProcessPreviousResults" ) );

			string[] orderGroupNumber = HL7Utility.ParseGetStringArray( this.Message.ORC[4], this.Message.Delimiters[1].ToString() );

			// An order should already exists in VBECS
			DataTable dtSearch = HL7AL.CprsOmgMessage.GetPatientOrderbyOrderGroupNumber(orderGroupNumber[0]);

			if ( dtSearch.Rows.Count > 0 )
			{
				DataTable dtPatientTest = new DataTable();
				DataRow drPatientTest = null;
				
				dtPatientTest.Columns.Add("PatientOrderGuid");
				dtPatientTest.Columns.Add("PatientGuid");
				dtPatientTest.Columns.Add("LabTestID");
				dtPatientTest.Columns.Add("TestResult");
				dtPatientTest.Columns.Add("ResultDateTime");
				dtPatientTest.Columns.Add("LabTestNormalRange");
				dtPatientTest.Columns.Add("LastUpdateUser");
				dtPatientTest.Columns.Add("LastUpdateFunctionId", typeof(int));
				dtPatientTest.Columns.Add("RecordStatusCode");
				dtPatientTest.Columns.Add("DivisionCode");

				drPatientTest = dtPatientTest.NewRow();

				drPatientTest["PatientOrderGuid"] = dtSearch.Rows[0]["PatientOrderGuid"];
				drPatientTest["PatientGuid"] = dtSearch.Rows[0]["PatientGuid"];
				drPatientTest["LabTestID"] = this.Message.UniversalServiceID[0];
				drPatientTest["TestResult"] = HL7Utility.ConvertString( this.Message.OBX[5] );
				drPatientTest["LabTestNormalRange"] = HL7Utility.ConvertString( this.Message.OBX[7] );
				drPatientTest["LastUpdateUser"] = LAST_UPDATE_USER_PARAMETER_VALUE;
				drPatientTest["LastUpdateFunctionId"]= Common.UpdateFunction.HL7CprsInterface;
				drPatientTest["RecordStatusCode"] = "H";
				drPatientTest["DivisionCode"] = dtSearch.Rows[0]["DivisionCode"];
				
				
				string resultDatetime = HL7Utility.ConvertString( this.Message.OBX[14] ).Substring(0, 12);
				if (resultDatetime != "")
				{
					try
					{
						DateTime resultDate = HL7DateFormat.ConvertHL7DateTime(resultDatetime);
						drPatientTest["ResultDateTime"] = resultDate;

					}
					catch(Exception)
					{
						// Indicates an invalid HL7 Date/Time format on the result date/time
						// from the value in the Lab file.
						throw( new HL7Exception("Missing or invalid result date.") );
					}
				}
				else
				{
					drPatientTest["ResultDateTime"] = null;
				}

				dtPatientTest.Rows.Add(drPatientTest);

				int retval = HL7AL.CprsOmgMessage.UpdatePatientPreviousResults(dtPatientTest);

				if (retval > 0)
				{
					return CprsOrgMessage.AckError( this.Message, AckCodes.AR, "Results Not Accepted" );

				}

				else
				{
					return CprsOrgMessage.AckError( this.Message, AckCodes.AA, "Results Accepted" );
				}
			}

			else
			{
				return CprsOrgMessage.AckError( this.Message, AckCodes.AR, "Unable to locate CPRS order group number" );

			}
		}

		#region [Properties]

		private HL7OmgMessage Message
		{
			get
			{
				return _message;
			}
		}

		#endregion
	}
}
